<?php


namespace Module\Wenku\Admin\Controller;


use Illuminate\Routing\Controller;
use ModStart\Admin\Auth\AdminPermission;
use ModStart\Admin\Concern\HasAdminQuickCRUD;
use ModStart\Admin\Layout\AdminCRUDBuilder;
use ModStart\Admin\Layout\AdminDialogPage;
use ModStart\Admin\Layout\AdminPage;
use ModStart\Core\Dao\ModelUtil;
use ModStart\Core\Exception\BizException;
use ModStart\Core\Input\InputPackage;
use ModStart\Core\Input\Request;
use ModStart\Core\Input\Response;
use ModStart\Core\Util\CRUDUtil;
use ModStart\Core\Util\FileUtil;
use ModStart\Core\Util\IdUtil;
use ModStart\Data\DataManager;
use ModStart\Field\AbstractField;
use ModStart\Field\AutoRenderedFieldValue;
use ModStart\Field\Type\FieldRenderMode;
use ModStart\Form\Form;
use ModStart\Grid\Displayer\ItemOperate;
use ModStart\Grid\GridFilter;
use ModStart\Layout\LayoutGrid;
use ModStart\Repository\Filter\RepositoryFilter;
use ModStart\Support\Concern\HasFields;
use ModStart\Widget\Box;
use ModStart\Widget\TextAjaxRequest;
use ModStart\Widget\TextDialogRequest;
use Module\Member\Util\MemberFieldUtil;
use Module\Member\Util\MemberMessageUtil;
use Module\Wenku\Core\WenkuDocSiteUrlBiz;
use Module\Wenku\Core\WenkuTagManagerBiz;
use Module\Wenku\Job\WenkuDocProcessJob;
use Module\Wenku\Model\WenkuDoc;
use Module\Wenku\Type\WenkuDocConvertStatus;
use Module\Wenku\Type\WenkuDocStatus;
use Module\Wenku\Util\DocCatUtil;
use Module\Wenku\Util\UrlUtil;
use Module\Wenku\Util\WenkuDocUtil;

class WenkuDocController extends Controller
{
    use HasAdminQuickCRUD;

    protected function crud(AdminCRUDBuilder $builder)
    {
        $builder
            ->init('wenku_doc')
            ->field(function ($builder) {
                /** @var HasFields $builder */
                $builder->id('id', 'ID');
                $builder->text('title', '标题')->hookRendering(function (AbstractField $field, $item, $index) {
                    switch ($field->renderMode()) {
                        case FieldRenderMode::DETAIL:
                        case FieldRenderMode::GRID:
                            return AutoRenderedFieldValue::make('<a href="' . modstart_web_url('doc/' . $item->hash) . '" target="_blank">' . htmlspecialchars($item->title) . '</a>');
                    }
                    return null;
                });
                $builder->type('status', '审核状态')->type(WenkuDocStatus::class);
                $builder->type('convertStatus', '转换状态')->type(WenkuDocConvertStatus::class);
                $builder->switch('isRecommend', '推荐')->optionsYesNo()->gridEditable(true);
                $builder->switch('anonymous', '匿名')->optionsYesNo()->gridEditable(true);
                $builder->switch('vipFreeDownloadEnable', '特权下载')->optionsYesNo()->gridEditable(true)->width(100)
                    ->tip('VIP用户有每日免费下载付费文档特权，开启后，用户上传的文档在VIP用户特权范围内可免费下载');
                $builder->select('categoryId', '分类')->optionModelTree('wenku_category');
                $builder->textarea('description', '描述');
                $builder->manyRelation('cats', '辅助分类')->relationTable('wenku_doc_cat_map', 'docId', 'catId')
                    ->groupTags(DocCatUtil::groupCatMap(-1), 'title', '_cats');
                $builder->tags('tags', '标签')->serializeAsColonSeparated();
                $builder->number('creditCost', '积分')->editable(true);
                $builder->decimal('moneyCost', '金额')->editable(true);
                $builder->number('viewCount', '浏览');
                $builder->number('downloadCount', '下载');
                $builder->number('previewPage', '预览页数')->editable(true);
                $builder->display('_count', '访问/评论/评分')
                    ->hookRendering(function (AbstractField $field, $item, $index) {
                        $html = [];
                        $html[] = intval($item->viewCount);
                        $html[] = intval($item->commentCount);
                        $html[] = intval($item->rateAvg);
                        return AutoRenderedFieldValue::make(join(' / ', $html));
                    })
                    ->listable(true)
                    ->showable(true);
                $builder->display('_countBase', '下载/点赞/收藏')
                    ->hookRendering(function (AbstractField $field, $item, $index) {
                        $template = '<span class="hover:tw-bg-gray-100 tw-px-1 tw-inline-block" data-tip-popover="真实">%d</span>+<span class="hover:tw-bg-gray-100 tw-px-1 tw-inline-block" data-tip-popover="虚拟">%d</span>';
                        $html = [];
                        $html[] = sprintf($template, intval($item->downloadCount), intval($item->downloadCountBase));
                        $html[] = sprintf($template, intval($item->likeCount), intval($item->likeCountBase));
                        $html[] = sprintf($template, intval($item->favoriteCount), intval($item->favoriteCountBase));
                        return AutoRenderedFieldValue::make(join(' / ', $html));
                    })
                    ->listable(true)
                    ->showable(true)
                    ->tip('前端显示数据为虚拟数据+真实数据');
                $builder->number('downloadCountBase', '下载虚拟数量')->editable(true)->showable(false)->listable(false);
                $builder->number('favoriteCountBase', '收藏虚拟数量')->editable(true)->showable(false)->listable(false);
                $builder->number('likeCountBase', '点赞虚拟数量')->editable(true)->showable(false)->listable(false);
                $builder->display('created_at', L('Created At'));
                $builder->display('updated_at', L('Updated At'));
            })
            ->repositoryFilter(function (RepositoryFilter $filter) {
                $filter->where('isDeleted', false);
            })
            ->gridFilter(function (GridFilter $filter) {
                $filter->eq('id', L('ID'));
                $filter->like('title', L('Title'));
                $filter->eq('status', '状态')->select(WenkuDocStatus::class);
                $filter->eq('convertStatus', '转换状态')->select(WenkuDocConvertStatus::class);
                $filter->has('categoryId', '分类')->cascaderModel('wenku_category');
            })
            ->canBatchSelect(true)
            ->canBatchDelete(true)
            ->batchOperatePrepend(
                join('', [
                    '<button class="btn" data-batch-dialog-operate="' . modstart_admin_url('wenku_doc/batch_edit') . '"><i class="iconfont icon-edit"></i> 批量编辑</button>',
                    '<button class="btn" data-batch-operate="' . modstart_admin_url('wenku_doc/verify_pass') . '"><i class="iconfont icon-check"></i> 审核通过</button>',
                    '<button class="btn" data-batch-operate="' . modstart_admin_url('wenku_doc/verify_reject') . '"><i class="iconfont icon-close"></i> 审核拒绝</button>',
                    '<button class="btn" data-batch-operate="' . modstart_admin_url('wenku_doc/convert_retry') . '"><i class="iconfont icon-play"></i> 转换重试</button>',
                    '<button class="btn" data-batch-operate="' . modstart_admin_url('wenku_doc/convert_retry_text') . '"><i class="iconfont icon-magic-wand"></i> 解析内容重试</button>',
                ])
            )
            ->hookItemOperateRendering(function (ItemOperate $itemOperate) {
                $itemOperate->getField()->gridFixed('right');
                $item = $itemOperate->item();
                $itemOperate->prepend(TextDialogRequest::primary('替换文件', action('\\' . __CLASS__ . '@replaceFile', ['_id' => $itemOperate->item()->id])));
                switch ($item->convertStatus) {
                    case WenkuDocConvertStatus::WAIT_PROCESS:
                    case WenkuDocConvertStatus::PROCESSING:
                    case WenkuDocConvertStatus::PROCESS_FAIL:
                        $itemOperate->prepend(TextAjaxRequest::warning('转换重试', action('\\' . __CLASS__ . '@convertRetry', ['_id' => $itemOperate->item()->id])));
                        break;
                    case WenkuDocConvertStatus::PROCESS_SUCCESS:
                        // $itemOperate->prepend(TextAjaxRequest::warning('自动描述', action('\\' . __CLASS__ . '@convertRetry', ['_id' => $itemOperate->item()->id])));
                        $itemOperate->prepend(TextAjaxRequest::warning('重新转换', action('\\' . __CLASS__ . '@convertRetry', ['_id' => $itemOperate->item()->id])));
                        break;
                }
                switch ($item->status) {
                    case WenkuDocStatus::VERIFYING:
                        $itemOperate->prepend(TextDialogRequest::make('warning', '审核', action('\\' . __CLASS__ . '@verify', ['_id' => $itemOperate->item()->id]))->size('big'));
                        // $itemOperate->prepend(TextAjaxRequest::danger('审核拒绝', action('\\' . __CLASS__ . '@verifyReject', ['_id' => $itemOperate->item()->id])));
                        break;
                }
            })
            ->hookSaving(function (Form $form) {
                $input = InputPackage::buildFromInput();
                $item = $form->item();
                if (modstart_module_enabled('TagManager')) {
                    $tags = $input->getTrimString('tags');
                    WenkuTagManagerBiz::updateTags($item->tags, $tags);
                }
                return Response::generateSuccess();
            })
            ->hookDeleted(function (Form $form) {
                $form->item()->each(function ($item) {
                    if (modstart_module_enabled('TagManager')) {
                        WenkuTagManagerBiz::deleteTags($item->tags);
                    }
                });
            })
            ->pageSizes([10, 100, 1000])
            ->canAdd(false)
            ->pageJumpEnable(true)
            ->title('文档管理');
    }

    public function batchEdit(AdminDialogPage $page)
    {
        $form = Form::make('');
        $form->showReset(false)->showSubmit(false)->formClass('wide');
        $form->switch('changeCategoryEnable', '修改分类')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->select('categoryId', '分类')->optionModelTree('wenku_category');
            });
        $form->switch('changeTagEnable', '修改标签')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->tags('tags', '标签')->serializeAsColonSeparated();
            });
        $form->switch('changeCatsEnable', '修改辅助分类')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->radio('changeCatsType', '修改类型')
                    ->options([
                        'replace' => '替换',
                        'append' => '追加',
                        'remove' => '移除',
                    ])
                    ->defaultValue('replace');
                $form->manyRelation('cats', '辅助分类')
                    ->groupTags(DocCatUtil::groupCatMap(-1), 'title', '_cats');
            });
        $form->switch('changeRecommendEnable', '修改推荐')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->switch('isRecommend', '推荐')->optionsYesNo();
            });
        $form->switch('changeCreditEnable', '修改购买积分')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->number('creditCost', '积分');
            });
        $form->switch('changeMoneyEnable', '修改购买金额')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->decimal('moneyCost', '金额');
            });
        $form->switch('viewCountEnable', '修改浏览量')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->number('viewCount', '浏览量');
            });
        $form->switch('previewPageEnable', '修改预览页数')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->number('previewPage', '预览页数');
            });
        $form->switch('vipFreeDownloadEnableEnable', '修改VIP免费下载')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->switch('vipFreeDownloadEnable', 'VIP免费下载')->optionsYesNo();
            });
        $form->switch('baseCountEnable', '修改虚拟数量')->optionsYesNo()
            ->when(true, function (Form $form) {
                $form->layoutGrid(function (LayoutGrid $layout) {
                    $layout->layoutColumn([4, 4], function ($builder) {
                        /** @var HasFields $builder */
                        $builder->number('downloadCountBase', '下载虚拟数量最小值');
                    });
                    $layout->layoutColumn([4, 4], function ($builder) {
                        /** @var HasFields $builder */
                        $builder->number('likeCountBase', '点赞虚拟数量最小值');
                    });
                    $layout->layoutColumn([4, 4], function ($builder) {
                        /** @var HasFields $builder */
                        $builder->number('favoriteCountBase', '收藏虚拟数量最小值');
                    });
                });
            });
        return $page->body($form)->pageTitle('批量编辑')->handleForm($form, function (Form $form) {
            $data = $form->dataForming();
            $ids = CRUDUtil::ids();
            BizException::throwsIfEmpty('ID为空', $ids);
            $update = [];
            if ($data['changeCategoryEnable']) {
                $update['categoryId'] = $data['categoryId'];
            }
            if ($data['changeTagEnable']) {
                $update['tags'] = $data['tags'];
            }
            if ($data['changeRecommendEnable']) {
                $update['isRecommend'] = $data['isRecommend'];
            }
            if ($data['changeCreditEnable']) {
                $update['creditCost'] = $data['creditCost'];
            }
            if ($data['changeMoneyEnable']) {
                $update['moneyCost'] = $data['moneyCost'];
            }
            if ($data['viewCountEnable']) {
                $update['viewCount'] = $data['viewCount'];
            }
            if ($data['previewPageEnable']) {
                $update['previewPage'] = $data['previewPage'];
            }
            if ($data['vipFreeDownloadEnableEnable']) {
                $update['vipFreeDownloadEnable'] = $data['vipFreeDownloadEnable'];
            }
            if ($data['baseCountEnable']) {
                $update['downloadCountBase'] = $data['downloadCountBase'];
                $update['likeCountBase'] = $data['likeCountBase'];
                $update['favoriteCountBase'] = $data['favoriteCountBase'];
            }
            AdminPermission::demoCheck();
            ModelUtil::transactionBegin();
            if (!empty($update)) {
                ModelUtil::model('wenku_doc')->whereIn('id', $ids)->update($update);
            }
            if ($data['changeCatsEnable']) {
                foreach ($ids as $id) {
                    $oldCats = ModelUtil::values('wenku_doc_cat_map', 'catId', ['docId' => $id]);
                    $cats = json_decode($data['cats'], true);
                    switch ($data['changeCatsType']) {
                        case 'replace':
                            $newCats = $cats;
                            break;
                        case 'append':
                            $newCats = array_merge($oldCats, $cats);
                            break;
                        case 'remove':
                            $newCats = array_filter($oldCats, function ($cat) use ($cats) {
                                return !in_array($cat, $cats);
                            });
                            break;
                    }
                    $newCats = array_unique($newCats);
                    ModelUtil::relationAssign('wenku_doc_cat_map', 'docId', $id, 'catId', $newCats);
                }
            }
            ModelUtil::transactionCommit();
            return Response::redirect(CRUDUtil::jsDialogCloseAndParentGridRefresh());
        });
    }

    public function delete()
    {
        AdminPermission::demoCheck();
        foreach (CRUDUtil::ids() as $id) {
            ModelUtil::update('wenku_doc', $id, ['isDeleted' => true]);
        }
        return Response::redirect(CRUDUtil::jsGridRefresh());
    }

    public function convertRetry()
    {
        AdminPermission::demoCheck();
        foreach (CRUDUtil::ids() as $id) {
            WenkuDocUtil::doConvertRetry($id);
        }
        return Response::json(0, '转换重试请求完成', null, CRUDUtil::jsGridRefresh());
    }

    public function convertRetryText()
    {
        AdminPermission::demoCheck();
        foreach (CRUDUtil::ids() as $id) {
            WenkuDocUtil::doConvertRetry($id, [
                'convertEnable' => false,
                'textEnable' => true,
            ]);
        }
        return Response::json(0, '转换重试请求完成', null, CRUDUtil::jsGridRefresh());
    }

    public function replaceFile(AdminDialogPage $page)
    {
        $doc = WenkuDocUtil::get(CRUDUtil::id());
        BizException::throwsIfEmpty('文档不存在', $doc);
        $form = Form::make('');
        $form->display('_from', '文档名称')->value($doc['title'])->addable(true);
        $form->fileTemp('file', '重新上传')->required()->autoSave(false);
        $form->showSubmit(false)->showReset(false);
        return $page
            ->pageTitle('替换文件')
            ->body($form)->handleForm($form, function (Form $form) use ($doc) {
                $data = $form->dataForming();
                $ret = DataManager::storeTempDataByPath($data['file']);
                if (Response::isError($ret)) {
                    return Response::generateError($ret['msg']);
                }
                $update = [];
                $update['fileUrl'] = $ret['data']['fullPath'];
                $update['fileSize'] = $ret['data']['data']['size'];
                $update['ext'] = FileUtil::extension($update['fileUrl']);
                $update['convertStatus'] = WenkuDocConvertStatus::WAIT_PROCESS;
                WenkuDocUtil::update($doc['id'], $update);
                WenkuDocProcessJob::create($doc['id'], Request::domainUrl());
                return Response::redirect(CRUDUtil::jsDialogCloseAndParentGridRefresh());
            });
    }

    public function verifyCheckTitle(AdminDialogPage $page)
    {
        $record = ModelUtil::get('wenku_doc', CRUDUtil::id());
        BizException::throwsIfEmpty('文档不存在', $record);
        $html = [];
        $records = ModelUtil::model('wenku_doc')
            ->where('title', $record['title'])
            ->where('id', '<>', $record['id'])
            ->get(['id', 'title', 'hash'])->toArray();
        foreach ($records as $r) {
            $html[] = "<div><a href='" . UrlUtil::doc($r) . "' target='_blank' rel='nofollow'><i class='iconfont icon-file'></i> " . htmlspecialchars($r['title']) . "</a></div>";
        }
        if (empty($records)) {
            $html[] = "<div><div class='ub-empty'><div class='icon'><i class='iconfont icon-empty-box'></i></div><div class='text'>没有相同标题文档</div></div></div>";
        }
        $html = "<div class='tw-leading-10'>" . join('', $html) . "</div>";
        return $page->pageTitle('查重《' . htmlspecialchars($record['title']) . '》')
            ->body(Box::make($html, '相同标题文档'));
    }

    public function verify(AdminDialogPage $page)
    {
        $record = ModelUtil::get('wenku_doc', CRUDUtil::id());
        BizException::throwsIfEmpty('文档不存在', $record);
        BizException::throwsIf('文档不是待审核状态', $record['status'] != WenkuDocStatus::VERIFYING);
        MemberFieldUtil::register();
        $form = Form::make();
        $form->layoutPanel('基础', function (Form $form) use ($record) {
            $form->adminMemberInfo('memberUserId', '上传者')->formAsDisplay(true);
            $duplicatedCount = ModelUtil::model('wenku_doc')
                ->where('title', $record['title'])
                ->where('id', '<>', $record['id'])
                ->count();
            $form->text('title', '标题')->help(
                '<a href="javascript:;" data-dialog-request="'
                . modstart_admin_url('wenku_doc/verify_check_title', ['_id' => $record['id']])
                . '"><i class="iconfont icon-search"></i> 查重找到 ' . $duplicatedCount . ' 个相同标题文档</a>'
            );
            $form->textarea('description', '描述');
            $form->select('categoryId', '分类')->optionModelTree('wenku_category');
            $form->tags('tags', '标签')->serializeAsColonSeparated();
            $form->manyRelation('cats', '辅助分类')->relationTable('wenku_doc_cat_map', 'docId', 'catId')
                ->groupTags(DocCatUtil::groupCatMap(-1), 'title', '_cats');
        });
        $form->layoutPanel('文档', function (Form $form) {
            $form->display('_preview', '文档预览')->hookRendering(function (AbstractField $field, $item, $index) {
                return AutoRenderedFieldValue::make('<a href="' . modstart_web_url('doc/' . $item->hash) . '" target="_blank">' . htmlspecialchars($item->title) . '</a>');
            })->addable(true);
        });
        $id = IdUtil::generate('Wenku');
        $form->layoutHtml(<<<HTML_FOOT
<script type="text/html" id="{$id}Content">
    <button type="button" class="btn btn-primary" data-verify-pass>审核通过</button>
    <button type="button" class="btn btn-danger" data-verify-reject>审核拒绝</button>
    <input class="form" placeholder="输入拒绝理由" data-verify-rejct-remark />
</script>
<script>
$(function(){
    $(".panel-dialog-foot").css('text-align','left').html($('#{$id}Content').html());
    var search = null;
    try {
        search = window.parent.__grids.get(0).lister.getParam().search;
    } catch(e) { }
    var request = function(data){
        MS.dialog.loadingOn();
        var formData = $('form').serializeJson();
        MS.api.post(window.location.href,Object.assign(data,{search:search},formData), function(res){
            MS.dialog.loadingOff();
            MS.api.defaultCallback(res);
        });
    };
    $(document).on('click','[data-verify-pass]',function(){
        request({verify:true});
    });
    $(document).on('click','[data-verify-reject]',function(){
        request({verify:false,remark:$('[data-verify-rejct-remark]').val()});
    });
});
</script>
HTML_FOOT
        );
        $form->showReset(false)->showSubmit(false);
        $form->item([
            'id' => $record['id'],
            'memberUserId' => $record['memberUserId'],
            'categoryId' => $record['categoryId'],
            'hash' => $record['hash'],
            'title' => $record['title'],
            'description' => $record['description'],
            'tags' => $record['tags'],
        ])->fillFields();
        return $page->pageTitle('审核《' . htmlspecialchars($record['title']) . '》')
            ->body($form)
            ->handleForm($form, function (Form $form) use ($record) {
                AdminPermission::demoCheck();
                $input = InputPackage::buildFromInput();
                $verify = $input->getBoolean('verify');
                $remark = $input->getTrimString('remark');
                $update = [];
                $update['title'] = $input->getTrimString('title');
                $update['description'] = $input->getTrimString('description');
                $update['categoryId'] = $input->getTrimString('categoryId');
                $update['tags'] = $input->getTrimString('tags');
                $cats = $input->getJson('cats');
                ModelUtil::relationAssign('wenku_doc_cat_map', 'docId', $record['id'], 'catId', $cats);
                if ($verify) {
                    ModelUtil::update('wenku_doc', $record['id'], $update);
                    WenkuDocUtil::docVerifyPass($record['id']);
                    WenkuDocSiteUrlBiz::update(UrlUtil::doc($record));
                } else {
                    BizException::throwsIfEmpty('拒绝理由不能为空', $remark);
                    ModelUtil::update('wenku_doc', $record['id'], $update);
                    WenkuDocUtil::doVerifyReject($record['id']);
                    MemberMessageUtil::sendTemplate($record['memberUserId'], 'wenkuDocVerifyReject', [
                        'doc' => $record,
                        'remark' => $remark,
                    ], 0, 'Wenku');
                }
                $search = $input->getArray('search');
                $filter = new GridFilter();
                $filter->eq('id', L('ID'));
                $filter->like('title', L('Title'));
                $filter->eq('status', '状态')->select(WenkuDocStatus::class);
                $filter->eq('convertStatus', '转换状态')->select(WenkuDocConvertStatus::class);
                $filter->has('categoryId', '分类')->cascaderModel('wenku_category');
                $filter->setSearch($search);
                $query = $filter->executeQueryConditions(WenkuDoc::query());
                $next = $query
                    ->where('status', WenkuDocStatus::VERIFYING)
                    ->where('id', '<', $record['id'])
                    ->orderBy('id', 'desc')
                    ->limit(1)->get()->toArray();
                if (!empty($next[0])) {
                    $url = CRUDUtil::adminUrlWithTab('wenku_doc/verify', [
                        '_id' => $next[0]['id'],
                    ]);
                    return Response::generate(0, null, null, $url);
                }
                return Response::generate(0, '全部处理完成', null, CRUDUtil::jsDialogCloseAndParentGridRefresh());
            });
    }

    public function verifyPass()
    {
        AdminPermission::demoCheck();
        foreach (CRUDUtil::ids() as $id) {
            WenkuDocUtil::docVerifyPass($id);
        }
        WenkuDocUtil::clearCache();
        return Response::json(0, null, null, CRUDUtil::jsGridRefresh());
    }

    public function verifyReject()
    {
        AdminPermission::demoCheck();
        foreach (CRUDUtil::ids() as $id) {
            WenkuDocUtil::doVerifyReject($id);
        }
        WenkuDocUtil::clearCache();
        return Response::json(0, null, null, CRUDUtil::jsGridRefresh());
    }

    public function verifyList(AdminPage $page)
    {
        $grid = $this->grid();
        $grid->disableCUD();
        $grid->canBatchDelete(true)->canDelete(true);
        $grid->repositoryFilter(function (RepositoryFilter $filter) {
            $filter->where('status', WenkuDocStatus::VERIFYING);
        });
        $grid->getGridFilter()->deleteFilterByColumn('status');
        CRUDUtil::registerGridResource($grid, '\\' . __CLASS__);
        if (Request::isPost()) {
            return $grid->request();
        }
        return $page->pageTitle('文档审核')->body($grid);
    }

    public function select(AdminDialogPage $page)
    {
        $grid = $this->grid();
        $grid->disableCUD();
        $grid->canMultiSelectItem(true);
        $grid->disableItemOperate();
        CRUDUtil::registerGridResource($grid, '\\' . __CLASS__);
        if (Request::isPost()) {
            return $grid->request();
        }
        return $page->pageTitle('选择文档')->body($grid);
    }

}
