<?php


namespace Module\Wenku\Api\Controller;


use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use ModStart\Admin\Auth\Admin;
use ModStart\Core\Assets\AssetsUtil;
use ModStart\Core\Dao\ModelUtil;
use ModStart\Core\Exception\BizException;
use ModStart\Core\Input\InputPackage;
use ModStart\Core\Input\Response;
use ModStart\Core\Type\TypeUtil;
use ModStart\Core\Util\ArrayUtil;
use ModStart\Core\Util\FileUtil;
use ModStart\Core\Util\FormatUtil;
use ModStart\Core\Util\IdUtil;
use ModStart\Core\Util\TagUtil;
use ModStart\Core\Util\TreeUtil;
use ModStart\Module\ModuleManager;
use Module\HotSearch\Util\HotSearchUtil;
use Module\Member\Auth\MemberUser;
use Module\Member\Auth\MemberVip;
use Module\Member\Support\MemberLoginCheck;
use Module\Member\Util\MemberCreditUtil;
use Module\Member\Util\MemberUtil;
use Module\Member\Util\MemberVipUtil;
use Module\MemberCert\Util\MemberCertUtil;
use Module\MemberFav\Util\MemberFavUtil;
use Module\MemberLike\Util\MemberLikeUtil;
use Module\PayCenter\Support\PayCenterPerform;
use Module\Vendor\Type\OrderStatus;
use Module\Vendor\Util\FileIconUtil;
use Module\Wenku\Core\WenkuDocGuestPayCenterBiz;
use Module\Wenku\Core\WenkuDocPayCenterBiz;
use Module\Wenku\Core\WenkuMemberFavBiz;
use Module\Wenku\Core\WenkuMemberLikeBiz;
use Module\Wenku\Event\WenkuDocDownloadedEvent;
use Module\Wenku\Model\DocDownload;
use Module\Wenku\Model\WenkuDoc;
use Module\Wenku\Model\WenkuDocCatMap;
use Module\Wenku\Provider\Convert\DocConverterProvider;
use Module\Wenku\Type\WenkuDocCatGroupViewMode;
use Module\Wenku\Type\WenkuDocPreviewType;
use Module\Wenku\Type\WenkuDocStatus;
use Module\Wenku\Util\DocCatUtil;
use Module\Wenku\Util\WenkuCategoryUtil;
use Module\Wenku\Util\WenkuDocDownloadUtil;
use Module\Wenku\Util\WenkuDocUtil;
use Module\Wenku\Util\WenkuDocVisitUtil;
use Module\Wenku\Util\WenkuDownloadUtil;
use Module\WenkuEnhance\Util\WenkuCartUtil;
use Module\WordSpliter\Util\WordSpliterUtil;

class WenkuDocController extends Controller implements MemberLoginCheck
{
    public static $memberLoginCheckIgnores = [
        'paginate', 'get', 'categoryDocs', 'downloadInfo',
        'buyGuest', 'buyGuestStatus', 'buyGuestGet',
        'downloadGuest', 'downloadGuestSubmit', 'downloadGuestFile',
    ];

    public function cats()
    {
        $map = DocCatUtil::groupCatMap();
        return Response::generateSuccessData($map);
    }

    public function categoryDocs()
    {
        $input = InputPackage::buildFromInput();
        $categoryId = $input->getInteger('categoryId');
        $limit = $input->getInteger('limit', 6);
        $docAll = $input->getBoolean('docAll', false);
        $categoryDocs = [];
        $categoryTree = WenkuCategoryUtil::categoryTree();
        if ($categoryId) {
            $categoryTree = array_filter($categoryTree, function ($item) use ($categoryId) {
                return $item['id'] == $categoryId;
            });
        }
        $categoryTree = array_filter($categoryTree, function ($o) {
            return !empty($o['showHome']);
        });
        foreach ($categoryTree as $item) {
            $one = [
                'categoryId' => $item['id'],
                'categoryTitle' => $item['title'],
                'categoryDesc' => $item['desc'],
            ];
            if ($docAll) {
                $one['docs'] = WenkuDocUtil::listCategoryLatest($item['id'], $limit);
            } else {
                $one['docs'] = WenkuDocUtil::listCategoryRecommendLatest($item['id'], $limit);
            }
            MemberUtil::mergeMemberUserBasics($one['docs']);
            $categoryDocs[] = $one;
        }
        return Response::generateSuccessData([
            'categoryDocs' => $categoryDocs,
        ]);
    }

    public function categories()
    {
        return Response::generateSuccessData([
            'categoryTree' => WenkuCategoryUtil::categoryTree(),
        ]);
    }

    public function recommends()
    {
        $input = InputPackage::buildFromInput();
        $categoryId = $input->getInteger('categoryId');
        $categoryIds = WenkuCategoryUtil::childIds($categoryId);
        $categoryIds[] = $categoryId;
        $records = ModelUtil::model('wenku_doc')
            ->where(['status' => WenkuDocStatus::VERIFY_PASS, 'isRecommend' => true, 'isDeleted' => false])
            ->whereIn('categoryId', $categoryIds)
            ->orderBy('id', 'desc')
            ->limit(10)
            ->get(WenkuDocUtil::BASIC_FIELDS)->toArray();
        ModelUtil::decodeRecordsJson($records, ['previewData', 'pagePreviews']);
        $records = WenkuDocUtil::buildForList($records);
        return Response::generateSuccessData([
            'records' => $records,
        ]);
    }

    public function latest()
    {
        $input = InputPackage::buildFromInput();
        $categoryId = $input->getInteger('categoryId');
        $categoryIds = WenkuCategoryUtil::childIds($categoryId);
        $categoryIds[] = $categoryId;
        $records = ModelUtil::model('wenku_doc')
            ->where(['status' => WenkuDocStatus::VERIFY_PASS, 'isDeleted' => false])
            ->whereIn('categoryId', $categoryIds)
            ->orderBy('id', 'desc')
            ->limit(10)
            ->get(WenkuDocUtil::BASIC_FIELDS)->toArray();
        ModelUtil::decodeRecordsJson($records, ['previewData', 'pagePreviews']);
        $records = WenkuDocUtil::buildForList($records);
        return Response::generateSuccessData([
            'records' => $records,
        ]);
    }

    public function free()
    {
        $input = InputPackage::buildFromInput();
        $categoryId = $input->getInteger('categoryId');
        $categoryIds = WenkuCategoryUtil::childIds($categoryId);
        $categoryIds[] = $categoryId;
        $records = ModelUtil::model('wenku_doc')
            ->where(['status' => WenkuDocStatus::VERIFY_PASS, 'isDeleted' => false])
            ->where(['moneyCost' => 0, 'creditCost' => 0])
            ->whereIn('categoryId', $categoryIds)
            ->orderBy('id', 'desc')
            ->limit(10)
            ->get(WenkuDocUtil::BASIC_FIELDS)->toArray();
        ModelUtil::decodeRecordsJson($records, ['previewData', 'pagePreviews']);
        $records = WenkuDocUtil::buildForList($records);
        return Response::generateSuccessData([
            'records' => $records,
        ]);
    }

    public function hottest()
    {
        $input = InputPackage::buildFromInput();
        $categoryId = $input->getInteger('categoryId');
        $categoryIds = WenkuCategoryUtil::childIds($categoryId);
        $categoryIds[] = $categoryId;
        $records = ModelUtil::model('wenku_doc')
            ->where(['status' => WenkuDocStatus::VERIFY_PASS, 'isDeleted' => false])
            ->whereIn('categoryId', $categoryIds)
            ->orderBy('viewCount', 'desc')
            ->limit(10)
            ->get(WenkuDocUtil::BASIC_FIELDS)->toArray();
        ModelUtil::decodeRecordsJson($records, ['previewData', 'pagePreviews']);
        $records = WenkuDocUtil::buildForList($records);
        return Response::generateSuccessData([
            'records' => $records,
        ]);
    }

    public function paginate()
    {
        $input = InputPackage::buildFromInput();
        $page = $input->getPage();
        $pageSize = $input->getPageSize(null, null, null, 24);
        $fields = [
            'id', 'hash', 'created_at', 'memberUserId', 'categoryId',
            'title', 'description', 'status', 'convertStatus', 'cover',
            'pageCount', 'viewCount', 'commentCount', 'scoreAvg',
            'downloadCount', 'favoriteCount', 'likeCount',
            'viewCountBase', 'downloadCountBase', 'favoriteCountBase', 'likeCountBase',
            'creditCost', 'moneyCost', 'rateAvg',
            'tags', 'previewData', 'ext', 'fileSize',
            'pagePreviews',
        ];
        $query = WenkuDoc::query()->where([
            'status' => WenkuDocStatus::VERIFY_PASS,
            'isDeleted' => false,
        ]);
        $filterText = [];
        $pageTitle = [];

        // 分类
        $categoryId = $input->getInteger('categoryId');
        $category = null;
        if ($categoryId) {
            $category = WenkuCategoryUtil::get($categoryId);
            BizException::throwsIfEmpty('分类不存在', $category);
            $filterText[] = $category['title'];
            $pageTitle[] = $category['title'];
            if ($category['pid']) {
                $parentCategory = WenkuCategoryUtil::get($category['pid']);
                $pageTitle[] = $parentCategory['title'];
            }
        }
        $categoryIds = [
            $categoryId
        ];
        $categories = WenkuCategoryUtil::all();
        $childrenIds = TreeUtil::nodesChildrenIds($categories, $categoryId);
        $categoryIds = array_merge($categoryIds, $childrenIds);
        $categoryChain = TreeUtil::nodesChainWithItems($categories, $categoryId);
        // print_r($categoryChain);exit();
        $query->whereIn('categoryId', $categoryIds);

        // 辅助分类
        $catIds = $input->getStringSeperatedIntegerArray('catIds', [], '_');
        if (!empty($catIds)) {
            foreach ($catIds as $catId) {
                $query->with('cats')->whereHas('cats', function ($q) use ($catId) {
                    $q->where(['catId' => $catId]);
                });
            }
            $catTitles = array_values(array_map(function ($o) {
                return $o['title'];
            }, DocCatUtil::listCatsByIds($catIds)));
            $filterText = array_merge($filterText, $catTitles);
        }
        $catGroups = DocCatUtil::groupCatMap($categoryId);
        foreach ($catGroups as $k => $catGroup) {
            $urlCatIds = ArrayUtil::sortNumber(ArrayUtil::removeAll($catIds, $catGroup['_catIds']));
            $catGroups[$k]['_urlCatIds'] = empty($urlCatIds) ? null : join('_', $urlCatIds);
            $catGroups[$k]['_isActive'] = ArrayUtil::hasAny($catGroup['_catIds'], $catIds);
            $catGroups[$k]['_active'] = null;
            foreach ($catGroup['_cats'] as $j => $cat) {
                $currentIds = ArrayUtil::add(ArrayUtil::removeAll($catIds, $catGroup['_catIds']), $cat['id']);
                $urlCatIds = ArrayUtil::sortNumber($currentIds);
                $catGroups[$k]['_cats'][$j]['_urlCatIds'] = empty($urlCatIds) ? null : join('_', $urlCatIds);
                $catGroups[$k]['_cats'][$j]['_isActive'] = in_array($cat['id'], $catIds);
                if ($catGroups[$k]['_cats'][$j]['_isActive']) {
                    $catGroups[$k]['_active'] = $cat;
                }
            }
        }

        // 格式
        $extList = modstart_config()->getArray('Wenku_FilterExt');
        $ext = $input->getTrimString('ext');
        if ($ext) {
            $exts = [];
            foreach ($extList as $item) {
                if ($item['k'] == $ext) {
                    $exts = explode(',', $item['v']);
                }
            }
            if (!empty($exts)) {
                $query->whereIn('ext', $exts);
                $filterText[] = strtoupper($ext);
            }
        }

        // 页码
        $pageFilter = $input->getInteger('pageFilter');
        $filterPages = modstart_config('Wenku_ListFilterPages', []);
        $pageFilters = [];
        if (!empty($filterPages)) {
            $prevValue = 0;
            foreach ($filterPages as $p) {
                $pageFilters[] = intval($prevValue) . '-' . intval($p);
                $prevValue = $p;
            }
            $pageFilters[] = intval($prevValue) . '以上';
            if ($pageFilter > 0) {
                if (isset($pageFilters[$pageFilter - 1])) {
                    if ($pageFilter == count($pageFilters)) {
                        $query->where('pageCount', '>=', $prevValue);
                    } else {
                        list($min, $max) = explode('-', $pageFilters[$pageFilter - 1]);
                        $query->whereBetween('pageCount', [$min, $max]);
                    }
                }
            }
        }

        // 免费、付费
        $feeType = $input->getTrimString('feeType');
        if ($feeType) {
            switch ($feeType) {
                case 'free':
                    $query->whereRaw("( ( creditCost = 0 OR creditCost IS NULL ) AND ( moneyCost = 0 OR moneyCost IS NULL ) )");
                    break;
                case 'fee':
                    $query->whereRaw("( creditCost > 0 OR moneyCost > 0 )");
                    break;
            }
        }

        // 关键词
        $keywords = $input->getTrimString('keywords');
        if ($keywords) {
            if (modstart_module_enabled('WordSpliter')) {
                $keywordsList = WordSpliterUtil::cut($keywords);
                $query->where(function ($q) use ($keywordsList, $keywords) {
                    $q->where(function ($qq) use ($keywordsList) {
                        foreach ($keywordsList as $item) {
                            $qq->where('title', 'like', '%' . $item . '%');
                        }
                    })->orWhere('tags', 'like', "%:$keywords:%");
                });
            } else {
                $query->where(function ($q) use ($keywords) {
                    $q->where('title', 'like', "%$keywords%")
                        ->orWhere('tags', 'like', "%:$keywords:%");
                });
            }
            if (modstart_module_enabled('HotSearch')) {
                HotSearchUtil::hit($keywords);
            }
            $filterText[] = $keywords;
        }

        $sort = $input->getTrimString('sort');
        switch ($sort) {
            case 'time':
                $query->orderBy('created_at', 'asc');
                break;
            case 'time_desc':
                $query->orderBy('created_at', 'desc');
                break;
            case 'hot':
                $query->orderBy('viewCount', 'desc');
                break;
            case 'download':
                $query->orderBy('downloadCount', 'desc');
                break;
            default:
                $query->orderBy('id', 'desc');
                break;
        }

        // 用户
        $memberUserId = $input->getInteger('memberUserId');
        if ($memberUserId) {
            $query = $query->where('memberUserId', $memberUserId);
        }

        $categoryCounts = null;
        $extCounts = null;

        if (modstart_config('Wenku_ListCountGroup', false)) {

            $queryClone = clone $query;
            // $queryClone->getQuery()->wheres = array_filter($queryClone->getQuery()->wheres, function ($item) {
            //     return !($item['type'] == 'In' && $item['column'] == 'categoryId');
            // });
            $queryClone = ModelUtil::queryRemoveCondition($queryClone, 'in', 'categoryId');
            // print_r(json_encode($queryClone->getQuery()->wheres, JSON_PRETTY_PRINT));
            // print_r(json_encode($queryClone->getQuery()->getRawBindings(), JSON_PRETTY_PRINT));
            // // exit();
            $counts = $queryClone->groupBy('categoryId')->select('categoryId', DB::raw('count(*) as count'))->get()->toArray();
            // print_r($counts);
            // exit();
            $map = array_column($counts, 'count', 'categoryId');
            $categoryCounts = [];
            foreach ($categories as $c) {
                $cid = $c['id'];
                $categoryCounts[$c['id']] = isset($map[$cid]) ? $map[$cid] : 0;
                $childrenIds = WenkuCategoryUtil::childIds($cid);
                // echo "$cid: ".join(',',$childrenIds)."\n";
                if (!empty($childrenIds)) {
                    foreach ($childrenIds as $childId) {
                        if (isset($map[$childId])) {
                            $categoryCounts[$cid] += $map[$childId];
                        }
                    }
                }
            }
            // print_r($categoryCounts);exit();
        }
        // print_r($extCounts);
        // exit();

        $result = $query->paginate($pageSize, $fields, 'page', $page)->toArray();

        $paginateData = [
            'total' => $result['total'],
            'records' => $result['data'],
        ];

        $categories = WenkuCategoryUtil::categoryTree();
        ModelUtil::decodeRecordsJson($paginateData['records'], ['previewData', 'pagePreviews']);
        DocCatUtil::mergeCats($paginateData['records']);
        foreach ($paginateData['records'] as $k => $record) {
            $paginateData['records'][$k]['tags'] = TagUtil::string2Array($record['tags']);
            $paginateData['records'][$k]['_categoryChain'] = TreeUtil::treeChain($categories, $record['categoryId']);
            $paginateData['records'][$k]['cover'] = WenkuDocUtil::cover($record);
            $paginateData['records'][$k]['_fileSize'] = FileUtil::formatByteSimple($record['fileSize']);
            $paginateData['records'][$k]['_time'] = date('m-d H:i', strtotime($record['created_at']));
            $paginateData['records'][$k]['_day'] = date('Y-m-d', strtotime($record['created_at']));
            $paginateData['records'][$k]['viewCount'] += intval($record['viewCountBase']);
            $paginateData['records'][$k]['downloadCount'] += intval($record['downloadCountBase']);
            $paginateData['records'][$k]['favoriteCount'] += intval($record['favoriteCountBase']);
            $paginateData['records'][$k]['likeCount'] += intval($record['likeCountBase']);
            $paginateData['records'][$k]['_icon'] = FileIconUtil::url($record['ext']);
            unset($paginateData['records'][$k]['previewData']);
            unset($paginateData['records'][$k]['pagePreviews']);
            unset($paginateData['records'][$k]['viewCountBase']);
            unset($paginateData['records'][$k]['downloadCountBase']);
            unset($paginateData['records'][$k]['favoriteCountBase']);
            unset($paginateData['records'][$k]['likeCountBase']);
        }
        MemberUtil::mergeMemberUserBasics($paginateData['records']);
        // print_r($paginateData['records']);exit();

        $catGroupsNormal = array_values(array_filter($catGroups, function ($o) {
            return $o['viewMode'] != WenkuDocCatGroupViewMode::MINI;
        }));
        $catGroupsMini = array_values(array_filter($catGroups, function ($o) {
            return $o['viewMode'] == WenkuDocCatGroupViewMode::MINI;
        }));

        return Response::generateSuccessData([
            'records' => $paginateData['records'],
            'total' => $paginateData['total'],
            'page' => $page,
            'pageSize' => $pageSize,
            'categoryId' => $categoryId,
            'category' => $category,
            'categoryChain' => $categoryChain,
            'categoryCounts' => $categoryCounts,
            'catGroupsNormal' => $catGroupsNormal,
            'catGroupsMini' => $catGroupsMini,
            'keywords' => $keywords,
            'catIds' => $catIds,
            'extList' => $extList,
            'ext' => $ext,
            'filterText' => $filterText,
            'sort' => $sort,
            'pageFilter' => $pageFilter,
            'pageFilters' => $pageFilters,
            'feeType' => $feeType,
        ]);
    }

    public function downloadInfo()
    {
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::getWithCache('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        $downloadInfo = null;
        if (!$downloadInfo && MemberUser::isLogin()) {
            $downloadInfo = ModelUtil::get('wenku_doc_download', [
                'docId' => $doc['id'],
                'memberUserId' => MemberUser::id(),
            ]);
            if ($downloadInfo) {
                $downloadInfo = [
                    'type' => 'member',
                    'memberUserId' => $downloadInfo['memberUserId'],
                ];
            }
        }
        if (!$downloadInfo) {
            // 其他授权下载
            // if (VisitIpLimitUtil::canVisit()) {
            //     $downloadInfo = [
            //         'type' => 'visitor',
            //         'visitor' => VisitIpLimitUtil::getVisitorByIp(),
            //     ];
            // }
        }
        return Response::generateSuccessData([
            'doc' => [
                'hash' => $doc['hash'],
                'title' => $doc['title'],
                'ext' => $doc['ext'],
                'creditCost' => $doc['creditCost'],
                'moneyCost' => $doc['moneyCost'],
                'vipFreeDownloadEnable' => boolval($doc['vipFreeDownloadEnable']),
            ],
            'isLogin' => MemberUser::isLogin(),
            'isVipEnable' => MemberVipUtil::isEnable(),
            'isMine' => MemberUser::isMine($doc['memberUserId']),
            'creditTotal' => MemberUser::isLogin() ? MemberCreditUtil::getTotal(MemberUser::id()) : 0,
            'creditName' => ModuleManager::getModuleConfig('Member', 'creditName', '积分'),
            'vipTitle' => MemberVip::get('title'),
            'vipIsDefault' => MemberVip::get('isDefault') ? true : false,
            'wenkuDiscount' => intval(MemberVip::get('wenkuDiscount', 100)),
            'vipFreeCount' => DocDownload::vipFreeCount(),
            'freeCount' => DocDownload::freeCount(),
            'guestEnable' => modstart_config('Wenku_GuestBuy', false),
            'guestEmailEnable' => modstart_config('Wenku_GuestBuyEmailEnable', false),
            'guestPhoneEnable' => modstart_config('Wenku_GuestBuyPhoneEnable', false),
            'downloadInfo' => $downloadInfo,
            'cartEnable' => modstart_module_enabled('WenkuEnhance') && modstart_config('Wenku_CartEnable', false),
            'cartExists' => modstart_module_enabled('WenkuEnhance')
                ? WenkuCartUtil::inCart(MemberUser::id(), $doc['id'])
                : false,
        ]);
    }

    public function get()
    {
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::getWithCache('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc, [
            'statusCode' => 404,
        ]);
        ModelUtil::decodeRecordJson($doc, ['pagePreviews', 'previewData']);
        if (!empty($doc['pagePreviews'])) {
            $doc['pagePreviews'] = AssetsUtil::fixFull($doc['pagePreviews']);
        }
        $doc['viewCount'] += intval($doc['viewCountBase']);
        $doc['downloadCount'] += intval($doc['downloadCountBase']);
        $doc['favoriteCount'] += intval($doc['favoriteCountBase']);
        $doc['likeCount'] += intval($doc['likeCountBase']);
        $doc = ArrayUtil::keepKeys($doc, [
            'id', 'hash', 'created_at', 'memberUserId', 'categoryId',
            'title', 'description', 'status', 'convertStatus', 'cover',
            'pageCount', 'viewCount', 'commentCount', 'scoreAvg',
            'downloadCount', 'favoriteCount', 'likeCount',
            'creditCost', 'moneyCost', 'rateAvg',
            'tags', 'pagePreviews', 'previewType', 'previewData', 'ext', 'provider',
            'previewPage', 'vipFreeDownloadEnable', 'anonymous',
        ]);
        $doc['cover'] = WenkuDocUtil::cover($doc);
        $doc['tags'] = TagUtil::string2Array($doc['tags']);
        $doc['_icon'] = FileIconUtil::url($doc['ext']);
        $categoryTree = WenkuCategoryUtil::categoryTree();
        $doc['_categoryChain'] = TreeUtil::treeChain($categoryTree, $doc['categoryId']);
        ModelUtil::update('wenku_doc', $doc['id'], [
            'viewCount' => $doc['viewCount'] + 1,
        ]);
        // 辅助分类
        $cats = [];
        $catIds = ModelUtil::values(WenkuDocCatMap::class, 'catId', ['docId' => $doc['id']]);
        if (!empty($catIds)) {
            $cats = DocCatUtil::listCatsByIds($catIds);
        }
        $doc['_cats'] = $cats;

        $doc['_memberUser'] = MemberUtil::getBasic($doc['memberUserId']);
        $doc['_memberUser']['_certType'] = null;
        if (modstart_module_enabled('MemberCert')) {
            $doc['_memberUser']['_certType'] = MemberCertUtil::getCertType($doc['memberUserId']);
        }
        if (MemberUser::isNotMine($doc['memberUserId'])) {
            if ($doc['status'] != WenkuDocStatus::VERIFY_PASS && Admin::isNotLogin()) {
                BizException::throws('文档当前状态为 ' . TypeUtil::name(WenkuDocStatus::class, $doc['status']));
            }
        }
        switch ($doc['previewType']) {
            case WenkuDocPreviewType::ZIP:
                if (!empty($doc['previewData']['tree'])) {
                    TreeUtil::treeMap($doc['previewData']['tree'], function (&$node) {
                        $node['icon'] = FileIconUtil::url(isset($node['type']) ? $node['type'] : '_DIR_');
                        if (isset($node['size'])) {
                            $node['sizeFormat'] = FileUtil::formatByteSimple($node['size']);
                        }
                    });
                }
                break;
            case WenkuDocPreviewType::IFRAME:
                $provider = DocConverterProvider::get($doc['provider']);
                if (!empty($provider)) {
                    $provider->init();
                    $ret = $provider->preview($doc);
                    if (Response::isSuccess($ret)) {
                        $doc['_iframeUrl'] = $ret['data']['url'];
                        // echo $doc['_iframeUrl'];exit();
                    }
                }
                break;
            case WenkuDocPreviewType::HTML_PAGE:
                $doc['_cssUrl'] = empty($doc['previewData']['css']) ? null : $doc['previewData']['css'];
                if (!empty($doc['_cssUrl'])) {
                    $doc['_cssUrl'] = AssetsUtil::fixFull($doc['_cssUrl']);
                }
                $doc['_htmlUrls'] = empty($doc['previewData']['html']) ? [] : $doc['previewData']['html'];
                $doc['_htmlContents'] = [];
                if (!empty($doc['_htmlUrls'])) {
                    $htmlPreviewPages = modstart_config('Wenku_HtmlPreviewPages', 3);
                    $doc['_htmlContents'] = Cache::remember('WenkuDoc:Html:' . $doc['id'] . ':' . $htmlPreviewPages,
                        60, function () use ($doc, $htmlPreviewPages) {
                            $ret = [];
                            foreach ($doc['_htmlUrls'] as $k => $v) {
                                if ($k >= $htmlPreviewPages) {
                                    $ret[$k] = [
                                        'type' => 'url',
                                        'url' => $v,
                                    ];
                                } else {
                                    $f = FileUtil::savePathToLocalTemp($v, 'html', true);
                                    $ret[$k] = [
                                        'type' => 'html',
                                        'html' => $f ? file_get_contents($f) : null,
                                    ];
                                    FileUtil::safeCleanLocalTemp($f);;
                                }
                            }
                            return $ret;
                        });
                    foreach ($doc['_htmlUrls'] as $k => $v) {
                        $doc['_htmlUrls'][$k] = AssetsUtil::fixFull($v);
                    }
                }
                break;
        }
        // print_r($doc);exit();
        if ($input->getBoolean('moreInfo', false)) {
            if (modstart_module_enabled('MemberLike')) {
                $doc['_hasLike'] = MemberLikeUtil::hasLike(MemberUser::id(), WenkuMemberLikeBiz::NAME, $doc['id']);
            }
            if (modstart_module_enabled('MemberFav')) {
                $doc['_hasFav'] = MemberFavUtil::hasFav(MemberUser::id(), WenkuMemberFavBiz::NAME, $doc['id']);
            }
            $doc['_maxPreviewPages'] = $doc['previewPage'];
            if (intval($doc['_maxPreviewPages']) <= 0) {
                $doc['_maxPreviewPages'] = modstart_config('Wenku_PreviewPageMax', 10);
            }
            $doc['_downloadTip'] = modstart_config('Wenku_DownloadTip');
        }
        if (MemberUser::isNotLogin()) {
            // 游客最大预览页数限制
            $maxPageForGuest = modstart_config('Wenku_PreviewPageMaxGuest', 10);
            if (count($doc['pagePreviews']) > $maxPageForGuest) {
                $doc['pagePreviews'] = array_slice($doc['pagePreviews'], 0, $maxPageForGuest);
            }
            if (isset($doc['previewData']['images'])) {
                $doc['previewData']['images'] = array_slice($doc['previewData']['images'], 0, $maxPageForGuest);
            }
        }

        WenkuDocVisitUtil::log($doc['id']);

        return Response::generateSuccessData([
            'doc' => $doc
        ]);
    }

    public function buyGuestStatus()
    {
        BizException::throwsIf('功能未开启', !modstart_config('Wenku_GuestBuy', false));
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::get('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        BizException::throwsIf('文档不需要购买', $doc['memberUserId'] == MemberUser::id());
        $guestId = $input->getTrimString('guestId');
        BizException::throwsIfEmpty('参数错误', $guestId);
        BizException::throwsIf('参数错误', strlen($guestId) != 32);
        $exists = ModelUtil::exists('wenku_doc_buy_order_guest', [
            'guestId' => $guestId,
            'docId' => $doc['id'],
            'status' => OrderStatus::COMPLETED,
        ]);
        // $exists = true;
        return Response::generateSuccessData([
            'status' => $exists ? 'bought' : 'not_buy',
        ]);
    }

    public function buyGuestGet()
    {
        BizException::throwsIf('功能未开启', !modstart_config('Wenku_GuestBuy', false));
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::get('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        $guestId = $input->getTrimString('guestId');
        BizException::throwsIfEmpty('参数错误', $guestId);
        BizException::throwsIf('参数错误', strlen($guestId) != 32);
        $record = ModelUtil::get('wenku_doc_buy_order_guest', [
            'guestId' => $guestId,
            'docId' => $doc['id'],
            'status' => OrderStatus::COMPLETED,
        ], ['*'], ['id', 'desc']);
        return Response::generateSuccessData([
            'doc' => ArrayUtil::keepKeys($doc, ['id', 'title', 'ext']),
            'record' => $record,
        ]);
    }

    public function buyGuest()
    {
        BizException::throwsIf('功能未开启', !modstart_config('Wenku_GuestBuy', false));
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::get('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        BizException::throwsIf('文档不需要购买', $doc['memberUserId'] == MemberUser::id());
        $guestId = $input->getTrimString('guestId');
        BizException::throwsIfEmpty('参数错误', $guestId);
        BizException::throwsIf('参数错误', strlen($guestId) != 32);
        $recvType = $input->getTrimString('recvType');
        $recvValue = $input->getTrimString('recvValue');
        $buyRecvEnable = (
            modstart_config('Wenku_GuestBuyEmailEnable', false)
            || modstart_config('Wenku_GuestBuyPhoneEnable', false)
        );
        if ($buyRecvEnable) {
            BizException::throwsIf('接收数据错误', !in_array($recvType, ['email', 'phone']));
            switch ($recvType) {
                case 'email':
                    BizException::throwsIfEmpty('请填写接收邮箱', $recvValue);
                    BizException::throwsIf('邮箱格式不正确', !FormatUtil::isEmail($recvValue));
                    break;
                case 'phone':
                    BizException::throwsIfEmpty('请填写接收手机', $recvValue);
                    BizException::throwsIf('手机格式不正确', !FormatUtil::isPhone($recvValue));
                    break;
            }
        }
        if (ModelUtil::exists('wenku_doc_buy_order_guest', [
            'guestId' => $guestId,
            'docId' => $doc['id'],
            'status' => OrderStatus::COMPLETED,
        ])) {
            return Response::generateError('已经购买');
        }
        /** @var PayCenterPerform $payCenterPerform */
        $payCenterPerform = app(PayCenterPerform::class);
        $moneyCost = $doc['moneyCost'];
        BizException::throwsIf('金额为0不能购买', $moneyCost < 0.01);
        $data = [
            'sn' => IdUtil::generateSN(),
            'guestId' => $guestId,
            'docId' => $doc['id'],
            'status' => OrderStatus::WAIT_PAY,
            'payFee' => $moneyCost,
        ];
        switch ($recvType) {
            case 'email':
                $data['email'] = $recvValue;
                break;
            case 'phone':
                $data['phone'] = $recvValue;
                break;
        }
        $order = ModelUtil::insert('wenku_doc_buy_order_guest', $data);
        $redirect = $input->getTrimString('redirect');
        if (empty($redirect)) {
            InputPackage::mergeToInput('redirect', modstart_web_url('doc/' . $doc['hash']));
        }
        return $payCenterPerform->performSubmitOrder(
            WenkuDocGuestPayCenterBiz::NAME,
            $order['id'],
            $order['payFee'],
            '购买《' . $doc['title'] . '》'
        );
    }

    public function buy()
    {
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::get('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        BizException::throwsIf('文档不需要购买', $doc['memberUserId'] == MemberUser::id());
        $where = [
            'docId' => $doc['id'],
            'memberUserId' => MemberUser::id(),
        ];
        BizException::throwsIf('文档不支持购买', $doc['moneyCost'] <= 0);
        BizException::throwsIf('文档已经购买', ModelUtil::exists('wenku_doc_download', $where));
        /** @var PayCenterPerform $payCenterPerform */
        $payCenterPerform = app(PayCenterPerform::class);
        $moneyCost = $doc['moneyCost'];
        if (MemberVipUtil::isEnable()) {
            $discount = max(0, min(MemberVip::get('wenkuDiscount', 100), 100));
            $moneyCost = bcdiv(bcmul($moneyCost, $discount, 2), 100, 2);
            if ($moneyCost <= 0.01) {
                $moneyCost = "0.01";
            }
        }
        $order = ModelUtil::insert('wenku_doc_buy_order', [
            'sn' => IdUtil::generateSN(),
            'memberUserId' => MemberUser::id(),
            'docId' => $doc['id'],
            'status' => OrderStatus::WAIT_PAY,
            'payFee' => $moneyCost,
        ]);
        $redirect = $input->getTrimString('redirect');
        if (empty($redirect)) {
            InputPackage::mergeToInput('redirect', modstart_web_url('doc/' . $doc['hash']));
        }
        return $payCenterPerform->performSubmitOrder(
            WenkuDocPayCenterBiz::NAME,
            $order['id'],
            $order['payFee'],
            '购买《' . $doc['title'] . '》'
        );
    }

    public function downloadVipSubmit()
    {
        $data = Response::tryGetData($this->downloadVip());
        return Response::generateSuccessData([
            'filename' => $data['filename'],
            'url' => action('\\' . __CLASS__ . '@downloadFile', ['hash' => $data['hash']])
        ]);
    }

    public function downloadGuestSubmit()
    {
        $data = Response::tryGetData($this->downloadGuest());
        return Response::generateSuccessData([
            'filename' => $data['filename'],
            'url' => action('\\' . __CLASS__ . '@downloadGuestFile', ['hash' => $data['hash']])
        ]);
    }

    public function downloadSubmit()
    {
        $data = Response::tryGetData($this->download());
        return Response::generateSuccessData([
            'filename' => $data['filename'],
            'url' => action('\\' . __CLASS__ . '@downloadFile', ['hash' => $data['hash']])
        ]);
    }

    public function downloadGuestFile($hash)
    {
        InputPackage::mergeToInput('hash', $hash);
        $data = Response::tryGetData($this->downloadGuest());
        $path = FileUtil::savePathToLocalTemp($data['url']);
        BizException::throwsIfEmpty('获取下载链接失败', $path);
        return Response::download($data['filename'], FileUtil::safeGetContent($path), [
            'Content-Type' => FileUtil::mime($data['ext']),
        ]);
    }

    public function downloadFile($hash)
    {
        InputPackage::mergeToInput('hash', $hash);
        $data = Response::tryGetData($this->download());
        $path = FileUtil::savePathToLocalTemp($data['url']);
        BizException::throwsIfEmpty('获取下载链接失败', $path);
        return Response::download($data['filename'], FileUtil::safeGetContent($path), [
            'Content-Type' => FileUtil::mime($data['ext']),
        ]);
    }

    public function downloadGuest()
    {
        BizException::throwsIf('功能未开启', !modstart_config('Wenku_GuestBuy', false));
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::get('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        BizException::throwsIf('文档不需要购买', $doc['memberUserId'] == MemberUser::id());
        $guestId = $input->getTrimString('guestId');
        BizException::throwsIfEmpty('参数错误', $guestId);
        BizException::throwsIf('参数错误', strlen($guestId) != 32);
        $exists = ModelUtil::exists('wenku_doc_buy_order_guest', [
            'guestId' => $guestId,
            'docId' => $doc['id'],
            'status' => OrderStatus::COMPLETED,
        ]);
        if (!$exists) {
            return Response::generateError('文档没有下载权限');
        }
        WenkuDocDownloadedEvent::fire($doc, null, $guestId);
        WenkuDocDownloadUtil::log($doc['id'], null, null, [
            'guestId' => $guestId
        ]);
        return Response::generateSuccessData(WenkuDownloadUtil::process($doc));
    }

    public function downloadVip()
    {
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::get('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        BizException::throwsIf('当前文档不支持VIP免费下载', !$doc['vipFreeDownloadEnable']);
        BizException::throwsIf('今日暂无免费额度', DocDownload::vipFreeCount() <= 0, [
            'view' => 'module::Wenku.View.pc.wenku.vipUpgrade'
        ]);
        ModelUtil::transactionBegin();
        $where = [
            'memberUserId' => MemberUser::id(),
            'docId' => $doc['id'],
        ];
        if (!ModelUtil::exists('wenku_doc_vip_free_download', $where)) {
            ModelUtil::insert('wenku_doc_vip_free_download', array_merge($where, [
                'day' => date('Y-m-d')
            ]));
        }
        $newDownload = false;
        if (!ModelUtil::exists('wenku_doc_download', $where)) {
            ModelUtil::insert('wenku_doc_download', $where);
            $newDownload = true;
        }
        if (DocDownload::vipFreeCount() < 0) {
            ModelUtil::transactionRollback();
            return Response::generateError('下载出现错误');
        }
        ModelUtil::transactionCommit();
        if ($newDownload) {
            WenkuDocUtil::updateDownloadCount($doc['id']);
        }
        return $this->download();
    }

    public function download()
    {
        $input = InputPackage::buildFromInput();
        $hash = $input->getTrimString('hash');
        $doc = ModelUtil::get('wenku_doc', ['hash' => $hash, 'isDeleted' => false,]);
        BizException::throwsIfEmpty('文档不存在', $doc);
        $where = [
            'docId' => $doc['id'],
            'memberUserId' => MemberUser::id(),
        ];
        if (ModelUtil::exists('wenku_doc_download', $where) || $doc['memberUserId'] == MemberUser::id()) {
        } else {
            if ($doc['creditCost']) {
                $creditCost = $doc['creditCost'];
                if (MemberVipUtil::isEnable()) {
                    $discount = max(0, min(MemberVip::get('wenkuDiscount', 100), 100));
                    $creditCost = ceil($creditCost * $discount / 100);
                    if ($doc['creditCost'] > 0) {
                        $creditCost = max($creditCost, 1);
                    }
                }
                $total = MemberCreditUtil::getTotal(MemberUser::id());
                BizException::throwsIf('积分不足', $total < $creditCost, [
                    'view' => 'module::Wenku.View.pc.wenku.creditUpgrade'
                ]);
                ModelUtil::transactionBegin();
                MemberCreditUtil::change(MemberUser::id(), -$creditCost, '下载文档《' . $doc['title'] . '》');
                MemberCreditUtil::change($doc['memberUserId'], +$doc['creditCost'], '文档下载《' . $doc['title'] . '》奖励');
                ModelUtil::insert('wenku_doc_download', array_merge($where, [
                    'creditCost' => $doc['creditCost'],
                ]));
                ModelUtil::transactionCommit();
                WenkuDocUtil::updateDownloadCount($doc['id']);
            } elseif ($doc['moneyCost'] > 0) {
                BizException::throws('请先购买文档');
            } else {
                BizException::throwsIf('今日暂无免费额度', DocDownload::freeCount() <= 0, [
                    'view' => 'module::Wenku.View.pc.wenku.vipUpgrade'
                ]);
                ModelUtil::insert('wenku_doc_download', array_merge($where, [
                    'creditCost' => $doc['creditCost'],
                ]));
                ModelUtil::insert('wenku_doc_free_download', [
                    'day' => date('Y-m-d'),
                    'docId' => $doc['id'],
                    'memberUserId' => MemberUser::id(),
                ]);
                WenkuDocUtil::updateDownloadCount($doc['id']);
            }
        }
        WenkuDocDownloadedEvent::fire($doc, MemberUser::user());
        WenkuDocDownloadUtil::log($doc['id']);
        return Response::generateSuccessData(WenkuDownloadUtil::process($doc));
    }
}
